#
# Copyright 2022 Ettus Research, a National Instruments Brand
#
# SPDX-License-Identifier: LGPL-3.0-or-later
#
# Description:
#   Timing constraints exclusive to X440. These should be used
#   in conjunction with ./common.xdc
#

###############################################################################
# DB GPIO
#   This interface is defined as system synchronous to pll_ref_clk.
#   Some timing constants in this section are declared in
#   <repo>/fpga/usrp3/top/x400/constraints/timing/shared_constants.sdc
###############################################################################

# In the current db_gpio_interface implementation, all GPIOs are received asynchronously
# by the DB, so we add a very relax constraint for these signals.
set db_gpio_min 0
set db_gpio_max [expr {$pll_ref_clk_period/2}]
set db_gpio_sync_max 3.500

# Set constraints for all ports.
set db_gpio_ports [get_ports {DB0_GPIO[*] DB1_GPIO[*]}]
set db_gpio_sync_ports [get_ports {DB0_GPIO[12] DB0_GPIO[13] DB1_GPIO[16] DB1_GPIO[17]}]

set_output_delay -clock [get_clocks pll_ref_clk] -min $db_gpio_min $db_gpio_ports
set_output_delay -clock [get_clocks pll_ref_clk] -max $db_gpio_max $db_gpio_ports
# Should just overwrite the setting for the sync clock ports that are also in db_gpio_ports
set_output_delay -clock [get_clocks ref_clk] -max $db_gpio_sync_max $db_gpio_sync_ports

set_input_delay -clock [get_clocks pll_ref_clk] -max $db_gpio_min $db_gpio_ports
set_input_delay -clock [get_clocks pll_ref_clk] -min $db_gpio_max $db_gpio_ports

# Use multi cycle path constraint to relax timing on these pins.
set_multicycle_path -setup 2 -from [get_pins db_gpio_gen[*].db_gpio_interface_i/ctrlport_to_i2c_inst/i2c_master/byte_controller/bit_controller/*_oen_reg/C]
set_multicycle_path -end -hold 7 -from [get_pins db_gpio_gen[*].db_gpio_interface_i/ctrlport_to_i2c_inst/i2c_master/byte_controller/bit_controller/*_oen_reg/C]
###############################################################################
# x440_ps_rfdc_bd
###############################################################################

# This property tells Vivado that we require the clocks generated by the RFDC MMCM to be well
# aligned. We have synchronous clock domain crossings between these clocks that can have
# large hold violations after placement due to uneven clock loading.
# The constraint CLOCK_DELAY_GROUP has to be applied on the clock nets.
# These nets are connected to the outputs of the clock buffers, which are driven by the MMCM.
# Search the nets begining from the MMCM outputs to find the clock buffers outputs.
set rfdc_mmcm [get_cells -hierarchical -filter { PRIMITIVE_TYPE == CLOCK.PLL.MMCME4_ADV && NAME =~ *rfdc/data_clock_mmcm* } ]
set mmcm_clkout_pins [get_pins -of_objects $rfdc_mmcm -regexp -filter {NAME =~ ".*CLKOUT[0-6]"}]
set mmcm_clkout_nets [get_nets -of_objects $mmcm_clkout_pins]
set clkbuf_in_pins [get_pins -leaf -of_objects $mmcm_clkout_nets -filter {DIRECTION == IN}]
set clkbuf_cells [get_cells -of_objects $clkbuf_in_pins]
set clkbuf_out_pins [get_pins -of_objects $clkbuf_cells -filter {DIRECTION == OUT}]
set clkbuf_out_nets [get_nets -of_objects $clkbuf_out_pins]
set_property CLOCK_DELAY_GROUP DataClkGroup $clkbuf_out_nets


# We treat rfdc_clk buffers as asynchronous, with knowledge that
# code clocked in this domain will be reset after this clocked is enabled. This
# will make timing easier to meet on these clock domains.
set_false_path -from [get_cells -hierarchical -regexp -filter {NAME =~ .*/rfdc/clock_gates_0/.*rEnableRfdc\dBufg.*_reg}]


###############################################################################
# SPLL SYSREF Capture
###############################################################################

# SYSREF is generated by the LMK04832 clocking chip (SPLL), which also produces
# the PLL reference clock (PRC) used to generate data clocks with a MMCM. Both
# SYSREF and PLL reference clock are directly fed into the RFSoC.
# The LMK04832 is configured to output all clocks synchronized to the
# rising edge of SYSREF. SYSREF is first captured by the FPGA fabric in the
# PRC clock domain (MMCM's PRC output) on the falling edge to meet setup and
# hold times. Afterwards it is captured with a double synchronizer and then
# transferred to the RFDC clock domain. Both SYSREF versions (PRC and RFDC)
# are used by downstream logic for sync purposes.
# The highest PRC frequency supported in MPM (64 MHz) is used for
# timing constraints.
# Set SYSREF LMK Delay to 0 ns as we assume rising edge alignment from the
# LMK04832 chip.
set sysref_lmk_delay 0
#
# These are the signals' lengths and corresponding delays (assuming 170 ps/in):
#   - SYSREF --> 5794 mils (5.794 inches) = 0.985 ns
#   - PRC    --> 5668 mils (5.668 inches) = 0.964 ns
#
# For min/max input delay calculations, it is assumed min prop. delay of 0 ns,
# which essentially over-constrains SYSREF.
#
# The max input delay is the latest that SYSREF may arrive w.r.t PRC, and it is
# calculated as follows:
#   Input delay (max) = SYSREF's LMK delay + SYSREF prop. delay (max)
#                       - PRC prop. delay (min)
set sysref_max_input_delay [expr {$sysref_lmk_delay + 0.985 - 0}]
#
# The min input delay is the earliest that SYSREF may arrive w.r.t PRC, and it
# is calculated as follows:
#   Input delay (min) = SYSREF's LMK delay + SYSREF prop. delay (min)
#                       - PRC prop. delay (min)
set sysref_min_input_delay [expr {$sysref_lmk_delay + 0 - 0.964}]

set_input_delay -clock pll_ref_clk -max $sysref_max_input_delay [get_ports {SYSREF_FABRIC_P}]
set_input_delay -clock pll_ref_clk -min $sysref_min_input_delay [get_ports {SYSREF_FABRIC_P}]

###############################################################################
# SPI to MB CPLD (PL)
#   This interface is defined as system synchronous to pll_ref_clk.
###############################################################################

# The output delays are chosen to allow a large time window of valid data for
# the MB CPLD logic.
set spi_min_out_delay  0.000
set spi_max_out_delay 11.000

# Set output constraints for all ports.
set spi_out_ports [get_ports {PL_CPLD_SCLK PL_CPLD_MOSI PL_CPLD_CS0_n PL_CPLD_CS1_n}]
set_output_delay -clock [get_clocks pll_ref_clk] -min $spi_min_out_delay $spi_out_ports
set_output_delay -clock [get_clocks pll_ref_clk] -max $spi_max_out_delay $spi_out_ports

# Force SPI registers into IOB.
set_property IOB TRUE [get_cells "cpld_interface_i/ctrlport_spi_master_i/simple_spi_core_64bit_i/sclk_reg"]
set_property IOB TRUE [get_cells "cpld_interface_i/ctrlport_spi_master_i/simple_spi_core_64bit_i/mosi_reg"]
